//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program.  If not, see http://www.gnu.org/licenses/.
// 

#include "RIPngRouting.h"

#include "IPv6InterfaceData.h"

Define_Module(RIPngRouting);

void RIPngRouting::showRoutingTable()
{
    RoutingTableIt it;
    RIPng::RoutingTableEntry *routingTableEntry;

    ev << routerText << endl;

    for (it = routingTable.begin(); it != routingTable.end(); it++)
    {
        routingTableEntry = (*it).second;
        ev << routingTableEntry->info() << endl;
    }
}

void RIPngRouting::initialize(int stage)
{
    if (stage != 3)
        return;

    // get the hostname
    cModule *containingMod = findContainingNode(this);
    if (!containingMod)
        hostName = "";
    else
        hostName = containingMod->getFullName();

    routerText = " (Router " + hostName + ") ";

    // access routing and interface table
    rt = RoutingTable6Access().get();
    ift = InterfaceTableAccess().get();
    // TODO: subscribe for changes such as NF_INTERFACE_STATE_CHANGED

    numSent = 0;
    numReceived = 0;
    WATCH(numSent);
    WATCH(numReceived);

    localPort = par("localPort");
    destPort = par("destPort");

    const char *RIPngAddressString = par("RIPngAddress");
    RIPngAddress = IPv6Address(RIPngAddressString);

    socket.setOutputGate(gate("udpOut"));
    socket.bind(localPort);
    setSocketOptions();

    // get deviceId
    deviceId = par("deviceId");

    // read the RIPng process configuration
    const char *RIPngConfigFileName = par("configFile");
    if (!loadConfigFromXML(RIPngConfigFileName))
        error("Error reading RIPng configuration from %s file", RIPngConfigFileName);

    simtime_t d = par("regularUpdateInterval").doubleValue();
    cMessage *regularUpdateTimerMsg = new cMessage("regularUpdateTimer");
    scheduleAt(d, regularUpdateTimerMsg);

    // TODO: send request instead of response
    sendRegularUpdateMessage(RIPngAddress, destPort);
}

bool RIPngRouting::loadConfigFromXML(const char *configFileName)
{
    // get router's config
    cXMLElement *routerNode = xmlParser::GetDevice("Router", deviceId, configFileName);
    if (routerNode == NULL)
    {
        ev << "No RIPng configuration found for this device" << routerText << endl;
        return true;
    }

    // interfaces config
    cXMLElement *interface;
    std::string RIPngInterfaceStatus;
      //get first router's interface
      interface = xmlParser::GetInterface(NULL, routerNode);
      while (interface != NULL)
      {// process all interfaces in config file
          const char *interfaceName = interface->getAttribute("name");
          RIPngInterfaceStatus = getInterfaceRIPngStatus(interface);
          if (RIPngInterfaceStatus == "enable")
              enableRIPngOnInterface(interfaceName);

          // process next interface
          interface = xmlParser::GetInterface(interface, NULL);
      }

    return true;
}

const char *RIPngRouting::getInterfaceRIPngStatus(cXMLElement *interface)
{
    if (interface == NULL)
        error("Error reading RIPng interface status, interface can't be NULL.");

    // get interface RIPng status
    const char *cRIPngStatus;
    cXMLElement* RIPngStatus = interface->getElementByPath("RIPngInterfaceStatus");
    if (RIPngStatus == NULL)
        return "disable";

    cRIPngStatus = RIPngStatus->getNodeValue();
    if (cRIPngStatus == NULL)
        return "disable";

    return cRIPngStatus;
}

void RIPngRouting::enableRIPngOnInterface(const char *interfaceName)
{
    ev << "Enabling RIPng on " << interfaceName << routerText << endl;

    //TODO: add RIPng multicast group on the interface
    InterfaceEntry *interface = ift->getInterfaceByName(interfaceName);
    int interfaceId = interface->getInterfaceId();
    RIPng::Interface *RIPngInterface = new RIPng::Interface(interfaceId);

    // add interface to local RIPng interface table
    addEnabledInterface(RIPngInterface);
    // add prefixes
    addPrefixesFromInterfaceToRT(RIPngInterface);
}

void RIPngRouting::addPrefixesFromInterfaceToRT(RIPng::Interface *RIPngInterface)
{
    // get interface form interfaceTable
    InterfaceEntry *interface = ift->getInterfaceById(RIPngInterface->getId());

    // get addresses on the interface and filter link-local and multicast addresses
    IPv6InterfaceData *ipv6data = interface->ipv6Data();
    int addressNum = ipv6data->getNumAddresses();
    IPv6Address addr;
    // TODO: how to get correct prefix and prefix length?
    int mask;
    RIPng::RoutingTableEntry *route;
    for (int i = 0; i < addressNum; i++)
    {
        addr = ipv6data->getAddress(i);
        mask = 64;
        if (!addr.isLinkLocal() && !addr.isMulticast())
        {
            // make directly connected route
            route = new RIPng::RoutingTableEntry(addr, mask);
            route->setInterfaceId(RIPngInterface->getId());
            route->setNextHop(IPv6Address::UNSPECIFIED_ADDRESS);  // means directly connected network
            addRoutingTableEntry(route);
        }
    }
}

void RIPngRouting::setSocketOptions()
{
    int timeToLive = par("timeToLive");
    if (timeToLive != -1)
        socket.setTimeToLive(timeToLive);

    const char *multicastInterface = par("multicastInterface");
    if (multicastInterface[0])
    {
        IInterfaceTable *ift = InterfaceTableAccess().get(this);
        InterfaceEntry *ie = ift->getInterfaceByName(multicastInterface);
        if (!ie)
            throw cRuntimeError("Wrong multicastInterface setting: no interface named \"%s\"", multicastInterface);
        socket.setMulticastOutputInterface(ie->getInterfaceId());
    }

    bool joinLocalMulticastGroups = par("joinLocalMulticastGroups");
    if (joinLocalMulticastGroups)
        socket.joinLocalMulticastGroups();
}

void RIPngRouting::handleMessage(cMessage *msg)
{
    if (msg->isSelfMessage())
    {// timers
        // send regular update message
        sendRegularUpdateMessage(RIPngAddress, destPort);
        // plan next regular update
        simtime_t d = simTime() + par("regularUpdateInterval").doubleValue();
        scheduleAt(d, msg);
    }
    else if (msg->getKind() == UDP_I_DATA)
    {// process incoming message
        processMessage(check_and_cast<RIPngMessage*> (msg));
    }
    else if (msg->getKind() == UDP_I_ERROR)
    {
        ev << "Ignoring UDP error report" << endl;
        delete msg;
    }
    else
    {
        error("Unrecognized message (%s)%s", msg->getClassName(), msg->getName());
    }

    if (ev.isGUI())
    {
        char buf[40];
        sprintf(buf, "rcvd: %d pks\nsent: %d pks", numReceived, numSent);
        getDisplayString().setTagArg("t", 0, buf);
    }

//    if (message->isSelfMessage()) {
//            handleTimer(check_and_cast<OSPFTimer*> (message));
//        } else if (dynamic_cast<ICMPMessage *>(message)) {
//            EV << "ICMP error received -- discarding\n";
//            delete message;
//        } else {
//            OSPFPacket* packet = check_and_cast<OSPFPacket*> (message);
//            EV << "Received packet: (" << packet->getClassName() << ")" << packet->getName() << "\n";
//            if (packet->getRouterID() == IPv4Address(router->getRouterID())) {
//                EV << "This packet is from ourselves, discarding.\n";
//                delete message;
//            } else {
//                processPacket(packet);
//            }
//        }
}

RIPngMessage *RIPngRouting::createMessage()
{
    char msgName[32] = "RIPngMessage";

    RIPngMessage *msg = new RIPngMessage(msgName);
    return msg;
}

void RIPngRouting::sendMessage(RIPngMessage *msg, IPv6Address &addr, int port, int outInterface)
{
    socket.sendTo(msg, addr, port, outInterface);
    numSent++;
}

void RIPngRouting::rebuildRoutingTable() {}

//-- INPUT PROCESSING
void RIPngRouting::processMessage(RIPngMessage *msg)
{
    EV << "Received packet: " << UDPSocket::getReceivedPacketInfo(msg) << endl;
    delete msg;
    numReceived++;
}

//-- RESPONSE PROCESSING
void RIPngRouting::processResponse() {}

// Unnecessary? virtual void checkMessageValidity()
void RIPngRouting::processRTEs() {}

// Unnecessary? virtual void checkRTE()
// Unnecessary? virtual void logBadRTE()
void RIPngRouting::updateRoutingTableEntry() {}

//-- REQUEST PROCESSING
void RIPngRouting::processRequest() {}

//-- TIMEOUTS
void RIPngRouting::routeTimeout() {}
void RIPngRouting::routeGarbageCollectionTimeout() {}
void RIPngRouting::triggeredUpdateTimer() {}
void RIPngRouting::startRouteDeletionProcess() {}
void RIPngRouting::deleteRoute() {}

//-- OUTPUT PROCESSING
void RIPngRouting::sendRegularUpdateMessage(IPv6Address &addr, int destPort)
{
    int numInterfaces = getEnabledInterfacesCount();

    RIPng::Interface *interface;
    RIPng::RoutingTableEntry *routingTableEntry;
    RoutingTableIt it;
    int size, interfaceId;
    std::vector<RIPngRTE> rtes;
    // sent update on every interface, where is enabled RIPng and that interface is not passive
    for (int i = 0; i < numInterfaces; i++)
    {
        interface = getEnabledInterface(i);
        interfaceId = interface->getId();
        RIPngMessage *msg = createMessage();
        rtes.clear();

        for (it = routingTable.begin(); it != routingTable.end(); it++)
        {
            routingTableEntry = (*it).second;
            if (routingTableEntry->getInterfaceId() == interfaceId)
                // split horizon
                continue;

            RIPngRTE rte;
            // create RTE for message to neighbor
            rte.setPrefixLen(routingTableEntry->getPrefixLength());
            rte.setIPv6Prefix(routingTableEntry->getDestPrefix());
            rte.setMetric(routingTableEntry->getMetric());
            // TODO: set tag to proper value
            rte.setRouteTag(0, 0);
            rte.setRouteTag(1, 0);

            rtes.push_back(rte);
        }

        size = rtes.size();
        msg->setCommand(RIPngResponse);
        msg->setRtesArraySize(size);
        // set RTEs to response
        for(int j = 0; j < size; j++)
            msg->setRtes(j, rtes[j]);

        sendMessage(msg, addr, destPort, interfaceId);
    }
}

void RIPngRouting::sendTriggeredUpdateMessage() {}

//-- GENERATING RESPONSE MESSAGES
void RIPngRouting::makeResponse(/*adresa na kterou se dela response - unicast, multicast(FF02::9)*/) {}
void RIPngRouting::makeTriggeredUpdate() {}
void RIPngRouting::clearRoutesChangeFlag() {}


